/*
 * Decompiled with CFR 0.152.
 */
package net.pms.network;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import java.net.InetAddress;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import net.pms.PMS;
import net.pms.configuration.UmsConfiguration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NetworkDeviceFilter {
    private static final Logger LOGGER = LoggerFactory.getLogger(NetworkDeviceFilter.class);
    private static final UmsConfiguration CONFIGURATION = PMS.getConfiguration();
    private static final String IP_FILTER_RULE_CHAR = "0123456789-.* ";
    private static final Pattern PATTERN = Pattern.compile("((\\d*)(-(\\d*))?)");
    private static final Map<String, Predicate> MATCHERS = new HashMap<String, Predicate>();
    private static final Set<String> LOGGED = new HashSet<String>();
    private static final Map<String, InetAddressSeen> LAST_SEEN = new HashMap<String, InetAddressSeen>();

    private NetworkDeviceFilter() {
        throw new IllegalStateException("Static class");
    }

    public static synchronized void reset() {
        MATCHERS.clear();
        LOGGED.clear();
        String rawFilter = CONFIGURATION.getNetworkDevicesFilter();
        if (rawFilter != null) {
            String[] rules;
            for (String r : rules = rawFilter.split("[,;]")) {
                Predicate p = NetworkDeviceFilter.parse(r);
                if (p == null) continue;
                MATCHERS.put(p.toString(), p);
            }
        }
    }

    public static synchronized boolean isAllowed(InetAddress addr) {
        return NetworkDeviceFilter.isAllowed(addr, true);
    }

    public static synchronized void setAllowed(String rule, boolean isAllowed) {
        boolean allowedByDefault;
        boolean bl = allowedByDefault = !CONFIGURATION.isNetworkDevicesBlockedByDefault();
        if (allowedByDefault == isAllowed) {
            MATCHERS.remove(rule);
        } else {
            Predicate p = NetworkDeviceFilter.parse(rule);
            if (p != null) {
                MATCHERS.put(p.toString(), p);
            }
        }
        CONFIGURATION.setNetworkDevicesFilter(NetworkDeviceFilter.getNormalizedFilter());
    }

    public static synchronized boolean getBlockedByDefault() {
        return CONFIGURATION.isNetworkDevicesBlockedByDefault();
    }

    public static synchronized void setBlockedByDefault(boolean value) {
        if (value != CONFIGURATION.isNetworkDevicesBlockedByDefault()) {
            CONFIGURATION.setNetworkDevicesFilter("");
            CONFIGURATION.setNetworkDevicesBlockedByDefault(value);
            NetworkDeviceFilter.reset();
        }
    }

    public static synchronized JsonArray getNetworkDevicesAsJsonArray() {
        Object device;
        JsonArray result = new JsonArray();
        boolean isAllowed = !PMS.getConfiguration().isNetworkDevicesBlockedByDefault();
        JsonObject rule = new JsonObject();
        rule.addProperty("name", "localhost");
        rule.addProperty("isDefault", true);
        rule.addProperty("isAllowed", true);
        JsonArray devices = new JsonArray();
        for (InetAddressSeen addrSeen : LAST_SEEN.values()) {
            if (!addrSeen.addr.isLoopbackAddress()) continue;
            device = new JsonObject();
            ((JsonObject)device).addProperty("hostName", addrSeen.addr.getHostName());
            ((JsonObject)device).addProperty("ipAddress", addrSeen.addr.getHostAddress());
            ((JsonObject)device).addProperty("lastSeen", addrSeen.seen);
            devices.add((JsonElement)device);
        }
        if (!devices.isEmpty()) {
            rule.add("devices", devices);
            result.add(rule);
        }
        for (Predicate predicate : MATCHERS.values()) {
            rule = new JsonObject();
            rule.addProperty("name", predicate.toString());
            rule.addProperty("isDefault", false);
            rule.addProperty("isAllowed", !isAllowed);
            devices = new JsonArray();
            for (InetAddressSeen addrSeen : LAST_SEEN.values()) {
                if (addrSeen.addr.isLoopbackAddress() || !predicate.match(addrSeen.addr)) continue;
                JsonObject device2 = new JsonObject();
                device2.addProperty("hostName", addrSeen.addr.getHostName());
                device2.addProperty("ipAddress", addrSeen.addr.getHostAddress());
                device2.addProperty("lastSeen", addrSeen.seen);
                devices.add(device2);
            }
            rule.add("devices", devices);
            result.add(rule);
        }
        for (InetAddressSeen addrSeen : LAST_SEEN.values()) {
            if (addrSeen.addr.isLoopbackAddress() || NetworkDeviceFilter.isAllowed(addrSeen.addr, false) != isAllowed) continue;
            rule = new JsonObject();
            rule.addProperty("name", addrSeen.addr.getHostAddress());
            rule.addProperty("isDefault", false);
            rule.addProperty("isAllowed", isAllowed);
            devices = new JsonArray();
            device = new JsonObject();
            ((JsonObject)device).addProperty("hostName", addrSeen.addr.getHostName());
            ((JsonObject)device).addProperty("ipAddress", addrSeen.addr.getHostAddress());
            ((JsonObject)device).addProperty("lastSeen", addrSeen.seen);
            devices.add((JsonElement)device);
            rule.add("devices", devices);
            result.add(rule);
        }
        return result;
    }

    private static synchronized boolean isAllowed(InetAddress addr, boolean logging) {
        boolean log;
        boolean bl = log = logging && NetworkDeviceFilter.isFirstDecision(addr);
        if (addr.isLoopbackAddress()) {
            if (log) {
                LOGGER.trace("IP Filter: Do not filter loopback address, access granted to {}", (Object)addr.getHostAddress());
            }
            return true;
        }
        boolean allowedByDefault = !CONFIGURATION.isNetworkDevicesBlockedByDefault();
        for (Predicate p : MATCHERS.values()) {
            if (!p.match(addr)) continue;
            if (allowedByDefault) {
                if (log) {
                    LOGGER.trace("IP Filter: Access denied to {} with rule: {} because it matched the denylist", (Object)addr.getHostAddress(), (Object)p);
                }
                return false;
            }
            if (log) {
                LOGGER.trace("IP Filter: Access granted to {} with rule: {} because it matched the allowlist", (Object)addr.getHostAddress(), (Object)p);
            }
            return true;
        }
        if (log) {
            if (allowedByDefault) {
                LOGGER.info("IP Filter: Access allowed to {}", (Object)addr.getHostName());
            } else {
                LOGGER.info("IP Filter: Access denied to {}", (Object)addr.getHostName());
            }
        }
        return allowedByDefault;
    }

    private static synchronized String getNormalizedFilter() {
        StringBuilder b = new StringBuilder();
        for (Predicate r : MATCHERS.values()) {
            if (b.length() > 0) {
                b.append(',');
            }
            b.append(r);
        }
        return b.toString();
    }

    private static synchronized boolean isFirstDecision(InetAddress addr) {
        String ip = addr.getHostAddress();
        LAST_SEEN.put(ip, new InetAddressSeen(addr));
        if (!LOGGED.contains(ip)) {
            LOGGED.add(ip);
            return true;
        }
        return false;
    }

    private static Predicate parse(String rule) {
        if ((rule = rule.trim()).length() == 0) {
            return null;
        }
        for (int i = 0; i < rule.length(); ++i) {
            if (IP_FILTER_RULE_CHAR.indexOf(rule.charAt(i)) != -1) continue;
            return new HostNamePredicate(rule);
        }
        String[] tags = rule.split("\\.");
        return new IpPredicate(tags);
    }

    static interface Predicate {
        public boolean match(InetAddress var1);
    }

    static class InetAddressSeen {
        InetAddress addr;
        long seen;

        public InetAddressSeen(InetAddress addr) {
            this.addr = addr;
            this.seen = System.currentTimeMillis();
        }
    }

    static class HostNamePredicate
    implements Predicate {
        String name;

        public HostNamePredicate(String n) {
            this.name = n.toLowerCase().trim();
        }

        @Override
        public boolean match(InetAddress addr) {
            return addr.getHostName().toLowerCase().contains(this.name);
        }

        public String toString() {
            return this.name;
        }
    }

    static class IpPredicate
    implements Predicate {
        List<ByteRule> byteRules;

        public IpPredicate(String[] tags) {
            this.byteRules = new ArrayList<ByteRule>(tags.length);
            for (String s : tags) {
                s = s.trim();
                this.byteRules.add(IpPredicate.parseTag(s));
            }
        }

        private static ByteRule parseTag(String s) {
            if ("*".equals(s)) {
                return new ByteRule(-1, -1);
            }
            Matcher matcher = PATTERN.matcher(s);
            if (matcher.matches()) {
                String start = matcher.group(2);
                String middle = matcher.group(3);
                String ending = matcher.group(4);
                if (IpPredicate.valid(start)) {
                    int x = Integer.parseInt(start);
                    if (IpPredicate.valid(ending)) {
                        return new ByteRule(x, Integer.parseInt(ending));
                    }
                    if (IpPredicate.valid(middle)) {
                        return new ByteRule(x, -1);
                    }
                    return new ByteRule(x, x);
                }
                if (IpPredicate.valid(ending)) {
                    return new ByteRule(-1, Integer.parseInt(ending));
                }
            }
            throw new IllegalArgumentException("Tag is not understood:" + s);
        }

        private static boolean valid(String s) {
            return s != null && s.length() > 0;
        }

        @Override
        public boolean match(InetAddress addr) {
            byte[] b = addr.getAddress();
            for (int i = 0; i < this.byteRules.size() && i < b.length; ++i) {
                int value;
                int n = value = b[i] < 0 ? b[i] + 256 : b[i];
                if (this.byteRules.get(i).match(value)) continue;
                return false;
            }
            return true;
        }

        public String toString() {
            StringBuilder b = new StringBuilder();
            for (ByteRule r : this.byteRules) {
                if (b.length() > 0) {
                    b.append('.');
                }
                b.append(r);
            }
            return b.toString();
        }

        static class ByteRule {
            int min;
            int max;

            public ByteRule(int a, int b) {
                this.min = a;
                this.max = b;
                if (b > -1 && a > -1 && b < a) {
                    this.max = a;
                    this.min = b;
                }
            }

            boolean match(int value) {
                if (this.min >= 0 && value < this.min) {
                    return false;
                }
                return this.max < 0 || value <= this.max;
            }

            public String toString() {
                StringBuilder builder = new StringBuilder();
                if (this.min >= 0) {
                    builder.append(this.min);
                    if (this.max > this.min) {
                        builder.append('-').append(this.max);
                    } else if (this.max < 0) {
                        builder.append('-');
                    }
                } else if (this.max >= 0) {
                    builder.append('-').append(this.max);
                } else {
                    builder.append('*');
                }
                return builder.toString();
            }
        }
    }
}

