#!/usr/bin/perl

# Receives a list of IP addresses and/or domain names
# on the command line or in a file via -i option
# Attempts to resolve them and outputs the results to stdout
# if -o filename opetion is specified, outputs a MagicTree XML
# to the specified file

use strict;
use Getopt::Std;
use XML::Writer;
use Socket;

my %opts;
getopts('i:o:', \%opts);

my($input, $xmloutput) =
    (
        $opts{'i'},
        $opts{'o'}
    );

if($input) {
    open(INPUT, $input) || die "Cannot open() file \"$input\": ". $!;
}

my $XMLOUTPUT;
if($xmloutput) {
    open($XMLOUTPUT, ">$xmloutput") || die "Canot open() file \"$xmloutput\" for writing: ". $!;
}

my @input;
if($input) { 
    @input = <INPUT>; # Reading input from file
    chomp @input;
} else {
    @input = @ARGV; # Reading input from command line
}

my $writer;

if($xmloutput) {
    $writer = new XML::Writer(OUTPUT => $XMLOUTPUT);
    $writer->startTag("magictree", "class"=>"MtBranchObject");
    $writer->startTag("testdata", "class"=>"MtBranchObject");
}

for my $i (@input) {
    if($i =~ /\d+\.\d+\.\d+\.\d+/) { # it's a dotted-decimal IP address
        if($xmloutput) {
            $writer->startTag("host", "status"=>"new");
            $writer->characters($i);
            $writer->endTag("host");
        }
        my $name = gethostbyaddr(inet_aton($i), AF_INET());
        if(!$name) {
            print STDERR "Cannot resolve address \"$i\": $!" ;
        } else {
            print "$i\t$name\n";
            if($xmloutput) {
                $writer->startTag("host", "status"=>"new");
                $writer->characters($i);
                $writer->startTag("hostname", "status"=>"new");
                $writer->characters($name);
                $writer->endTag("hostname");
                $writer->endTag("host");    
            }
        }
    } else {  # It is not an IP address, so let's assume it is a DNS name
        if($xmloutput) {
            $writer->startTag("fqdn", "status"=>"new");
            $writer->characters($i);
            $writer->endTag("fqdn");
        } 
        my @addresses = gethostbyname($i);
        if(!@addresses) {
            print STDERR "Cannot resolve name \"$i\": $!" ;
        } else {
            @addresses = map { inet_ntoa($_) } @addresses[4 .. $#addresses];
            # Now @addresses is a list of dotted=decimal strings
            for my $a (@addresses) {
                print "$i\t$a\n";
                if($xmloutput) {
                    $writer->startTag("host", "status"=>"new");
                    $writer->characters($a);
                    $writer->startTag("hostname", "status"=>"new");
                    $writer->characters($i);
                    $writer->endTag("hostname");
                    $writer->endTag("host");  
                }
            }
        }
    }
}    

if($xmloutput) {
    $writer->endTag("testdata", "class"=>"MtBranchObject");
    $writer->endTag("magictree", "class"=>"MtBranchObject");
}
