#!/usr/bin/perl -w

## TODO: 1) limit by banner size 2) limit number of simultaneous connections
## On stdin this script expects tab-separated list of port-host pairs
## On stderr this script produces debugging information
## On stdout this script dumps XML

use strict;

use FindBin qw($Bin);
use lib "$Bin/../lib";

use IO::Socket::INET;
use IO::Select;
use Errno qw(EAGAIN EBADF);
use String::Escape qw/qprintable/;
#use Data::Dumper;
use MagicTree::XMLWriter;

my $force_close_after = 60;	## whatever happens, do not spend more than so many seconds per port
my $send_crlf_after = 5;	## send CR/LF after so many seconds of inactivity
my $close_after_idle = 5;	## give up after so many seconds of inactivity

my $bufsize = 1024;

## --- prepare the list of targets in @peers
my @peers;
for(my $nline=1; <STDIN>; $nline++) {
	die "malformed input line $nline: '$_'" unless /^(\S+)\t(\S+)$/;
	push @peers, "$2:$1";
}

## --- initiate the connections
my $sel_connecting = new IO::Select;
my $sel_connected = new IO::Select;

my @banners;

foreach my $peeraddr (@peers) {
	my $s = new IO::Socket::INET(PeerAddr => $peeraddr, Blocking => 0) || die;

	${*$s}{'peeraddr'} = $peeraddr;
	$sel_connecting->add($s);
}

while($sel_connecting->handles || $sel_connected->handles) {
	my ($readable, $writeable) = IO::Select::select($sel_connected, $sel_connecting, undef, 1);

	my $now = time;
	for my $s (@$writeable) {
		$sel_connecting->remove($s);

		${*$s}{'banner'} = "";
		${*$s}{'force_close_at'} = $now + $force_close_after;
		${*$s}{'send_crlf_at'} = $now + $send_crlf_after;
		$sel_connected->add($s);

		printf STDERR "* connected to '%s'\n", ${*$s}{'peeraddr'};
	}

	for my $s (@$readable) {
		my $buf = "";
		my $nread = $s->read($buf, $bufsize);
		if (defined $nread) {
			if ($nread == 0) {
				## eof
				$sel_connected->remove($s);
				$s->close();

				push @banners, $s;

				printf STDERR "* connection closed by '%s'\n", ${*$s}{'peeraddr'};
			} else {
				## got some new data, keep it
				${*$s}{'banner'} .= $buf;
				${*$s}{'close_at'} = time() + $close_after_idle;
				#undef ${*$s}{'send_crlf_at'};

				printf STDERR "* got %d bytes from %s: %s\n",
					length($buf), ${*$s}{'peeraddr'}, qprintable($buf);
			}
		} else {
			## error
			my $err = $!;
			if ($err != EAGAIN) {
				## hard failure
				$sel_connected->remove($s);
				$s->close();

				unless ($err == EBADF) {
					${*$s}{'error'} = $err;
				}

				push @banners, $s;

				printf STDERR "* error reading from '%s': %s\n", ${*$s}{'peeraddr'}, $err;
			} else {
				## should not happen, take no actions
				warn "Error: no data read from a readable socket\n";
			}
		}
	}

	for my $s ($sel_connected->handles) {
		if ($now > ${*$s}{'force_close_at'}) {
			$sel_connected->remove($s);
			$s->close();

			push @banners, $s;

			printf STDERR "* connection with '%s' is forcefully closed\n", ${*$s}{'peeraddr'};
		} elsif (defined(${*$s}{'send_crlf_at'}) && ($now > ${*$s}{'send_crlf_at'})) {
			#$s->print("\r\n"); $s->flush();
			$s->print("\r\n"); $s->flush();
			undef ${*$s}{'send_crlf_at'};
			${*$s}{'close_at'} = $now + $close_after_idle;

			printf STDERR "* CRLF sent to '%s'\n", ${*$s}{'peeraddr'};
		} elsif (defined(${*$s}{'close_at'}) && ($now > ${*$s}{'close_at'})) {
			$sel_connected->remove($s);
			$s->close();

			push @banners, $s;

			printf STDERR "* connection with '%s' is closed by idle timeout\n", ${*$s}{'peeraddr'};
		}
	}
}

## --- dump the results in XML ---

my $writer = new MagicTree::XMLWriter();
$writer->startTestdataDocument();

sub writeBanner($$) {
        my $writer = shift;
        my $s = shift;

        die("Cannot parse peeraddr '" . ${*$s}{'peeraddr'} . "'")
                unless ${*$s}{'peeraddr'} =~ /^([^:]*):([^:]+)$/;
        my $host = $1;
        my $port = $2;
        my $bannerText = ${*$s}{'banner'};

        $writer->startTag("host");
        $writer->characters($host);

        $writer->startTag("ipproto");
        $writer->characters("tcp");

        $writer->startTag("port");
        $writer->characters($port);

	if (length ${*$s}{'banner'} > 0) {
		## there was no error or banner is not empty
        	$writer->startTag("banner");
        	$writer->characters(${*$s}{'banner'});
        	$writer->endTag("banner");
	}

	if (defined ${*$s}{'error'} && length ${*$s}{'error'} > 0) {
		## there was an error
        	$writer->startTag("banner_error");
        	$writer->characters(${*$s}{'error'});
        	$writer->endTag("banner_error");
	}

        $writer->endTag("port");
        $writer->endTag("ipproto");
        $writer->endTag("host");
}

for my $banner (@banners) {
        writeBanner($writer, $banner);
}

$writer->endTestdataDocument();

