"use strict";
var _ = require('underscore');
var JSFile = require('./js-file');
var logger = require('../logger');
var LeenaContext = (function () {
    function LeenaContext(leenaConfig) {
        this.leenaConfig = leenaConfig;
        this.jsFiles = [];
    }
    LeenaContext.prototype.setChromeServer = function (chromeServer) {
        this.chromeServer = chromeServer;
    };
    LeenaContext.prototype.handleProperty = function (event, fileInfo) {
        switch (event) {
            case 'add':
                try {
                    if (fileInfo.fileToTest) {
                        this.add(fileInfo);
                    }
                }
                catch (e) {
                    throw e;
                }
                break;
            case 'change':
                try {
                    if (fileInfo.fileToTest) {
                        this.update(fileInfo);
                        logger.info('Updating instance of "' + fileInfo.pathFile + '"');
                    }
                }
                catch (e) {
                    throw e;
                }
                break;
            case 'unlink':
                try {
                    if (fileInfo.fileToTest) {
                        this.delete(fileInfo);
                        logger.info('Deleting instance of "' + fileInfo.pathFile + '"');
                    }
                }
                catch (e) {
                    throw e;
                }
                break;
            case 'unlinkDir':
                break;
        }
    };
    LeenaContext.prototype.executeFunction = function (functionName, cb) {
        var found = false;
        for (var k = 0; k < this.jsFiles.length; k++) {
            if (this.jsFiles[k].containsFunction(functionName)) {
                this.jsFiles[k].executeFunction(functionName, cb);
                found = true;
                break;
            }
        }
        if (!found) {
            cb(new Error('Unable to execute function "' + functionName + '". It does not exist'), null);
        }
    };
    LeenaContext.prototype.updateFunctionInstance = function (functionName, functionI) {
        var found = false;
        for (var k = 0; k < this.jsFiles.length; k++) {
            if (this.jsFiles[k].containsFunction(functionName)) {
                this.jsFiles[k].updateFunctionInstance(functionName, functionI);
                found = true;
                break;
            }
        }
        if (!found) {
            throw new Error('Unable to update function "' + functionName + '". It does not exist');
        }
    };
    LeenaContext.prototype.getFunction = function (functionName) {
        var index = this.getFunctionIndex(functionName);
        if (index === -1) {
            throw new Error('Unable to get function instance. Function "' + functionName + '" does not exist');
        }
        else if (index < 0 || index >= this.jsFiles.length) {
            throw new Error('Unable to get function instance of function "' + functionName + '". Index out of range');
        }
        return this.jsFiles[index].getFunctionInstance(functionName);
    };
    LeenaContext.prototype.getFunctionIndex = function (functionName) {
        for (var k = 0; k < this.jsFiles.length; k++) {
            if (this.jsFiles[k].containsFunction(functionName)) {
                return k;
            }
        }
        return -1;
    };
    LeenaContext.prototype.add = function (fileInfo) {
        var index = this.get(fileInfo.pathTempFile);
        if (index === -1) {
            var jsFile;
            var that = this;
            jsFile = new JSFile(fileInfo, this.leenaConfig, function (err, res) {
                if (err) {
                    throw err;
                }
                else if (res) {
                    that.jsFiles.push(jsFile);
                    that.chromeServer.updateContext(that);
                }
            });
        }
        else {
            throw new Error('Unable to add file "' + fileInfo.pathTempFile + '" in the context. It already exists');
        }
    };
    LeenaContext.prototype.delete = function (fileInfo) {
        var index = this.get(fileInfo.pathTempFile);
        if (index !== -1 && index <= this.jsFiles.length) {
            this.jsFiles.splice(index, 1);
        }
        else {
            throw new Error('Unable to delete file "' + fileInfo.pathTempFile + '" in the context');
        }
    };
    LeenaContext.prototype.update = function (fileInfo) {
        var index = this.get(fileInfo.pathTempFile);
        if (index !== -1 && index <= this.jsFiles.length) {
            this.jsFiles[index].update();
        }
        else {
            throw new Error('Unable to update file "' + fileInfo.pathTempFile + '" in the context');
        }
    };
    LeenaContext.prototype.get = function (pathTempFile) {
        if (!_.isArray(this.jsFiles)) {
            return -1;
        }
        for (var k = 0, lengthJF = this.jsFiles.length; k < lengthJF; k++) {
            if (this.jsFiles[k].pathTempFile === pathTempFile) {
                return k;
            }
        }
        return -1;
    };
    return LeenaContext;
}());
module.exports = LeenaContext;
