/*
  Copyright (C) 2012-2013 Yusuke Suzuki <utatane.tea@gmail.com>

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
  ARE DISCLAIMED. IN NO EVENT SHALL <COPYRIGHT HOLDER> BE LIABLE FOR ANY
  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*jslint bitwise:true */
/*global module:true, require:true*/
(function () {
    'use strict';

    var Name, Syntax, common, escope, evaluator, modified;

    Name = 'remove-unreachable-branch';
    escope = require('escope');
    common = require('../common');
    evaluator = require('../evaluator');
    Syntax = common.Syntax;

    function handleIfStatement(func, node) {
        var test, body, decl;
        test = evaluator.booleanCondition(node.test);
        if (!node.alternate) {
            if (typeof test === 'boolean') {
                modified = true;
                body = [];

                if (test) {
                    body.push(common.moveLocation(node.test, {
                        type: Syntax.ExpressionStatement,
                        expression: node.test
                    }), node.consequent);
                    return {
                        type: Syntax.BlockStatement,
                        body: body
                    };
                } else {
                    decl = common.delegateVariableDeclarations(node.consequent, func);
                    if (decl) {
                        body.push(decl);
                    }
                    body.push(common.moveLocation(node.test, {
                        type: Syntax.ExpressionStatement,
                        expression: node.test
                    }));
                    return {
                        type: Syntax.BlockStatement,
                        body: body
                    };
                }
            }
        } else {
            if (typeof test === 'boolean') {
                modified = true;
                body = [];

                if (test) {
                    decl = common.delegateVariableDeclarations(node.alternate, func);
                    if (decl) {
                        body.push(decl);
                    }
                    body.push(common.moveLocation(node.test, {
                        type: Syntax.ExpressionStatement,
                        expression: node.test
                    }), node.consequent);
                    return {
                        type: Syntax.BlockStatement,
                        body: body
                    };
                } else {
                    decl = common.delegateVariableDeclarations(node.consequent, func);
                    if (decl) {
                        body.push(decl);
                    }
                    body.push(common.moveLocation(node.test, {
                        type: Syntax.ExpressionStatement,
                        expression: node.test
                    }), node.alternate);
                    return {
                        type: Syntax.BlockStatement,
                        body: body
                    };
                }
            }
        }
    }

    function handleLogicalExpression(func, node) {
        var test;
        test = evaluator.booleanCondition(node.left);
        if (typeof test === 'boolean') {
            modified = true;
            if (test) {
                if (node.operator === '&&') {
                    return common.moveLocation(node, {
                        type: Syntax.SequenceExpression,
                        expressions: [ node.left, node.right ]
                    });
                } else {
                    return node.left;
                }
            } else {
                if (node.operator === '&&') {
                    return node.left;
                } else {
                    return common.moveLocation(node, {
                        type: Syntax.SequenceExpression,
                        expressions: [ node.left, node.right ]
                    });
                }
            }
        }
    }

    function handleConditionalExpression(func, node) {
        var test;
        test = evaluator.booleanCondition(node.test);
        if (typeof test === 'boolean') {
            modified = true;
            if (test) {
                return common.moveLocation(node, {
                    type: Syntax.SequenceExpression,
                    expressions: [ node.test, node.consequent ]
                });
            } else {
                return common.moveLocation(node, {
                    type: Syntax.SequenceExpression,
                    expressions: [ node.test, node.alternate ]
                });
            }
        }
    }

    function removeUnreachableBranch(tree, options) {
        var result, stack;

        result = options.get('destructive', { pathName: Name }) ? tree : common.deepCopy(tree);
        modified = false;
        stack = [];

        result = common.replace(result, {
            enter: function enter(node) {
                var func;

                if (escope.Scope.isVariableScopeRequired(node)) {
                    stack.push(node);
                    return;
                }
                func = common.Array.last(stack);

                switch (node.type) {
                case Syntax.IfStatement:
                    return handleIfStatement(func, node);

                case Syntax.LogicalExpression:
                    return handleLogicalExpression(func, node);

                case Syntax.ConditionalExpression:
                    return handleConditionalExpression(func, node);
                }
            },
            leave: function leave(node) {
                if (escope.Scope.isVariableScopeRequired(node)) {
                    stack.pop();
                }
            }
        });

        return {
            result: result,
            modified: modified
        };
    }

    removeUnreachableBranch.passName = Name;
    module.exports = removeUnreachableBranch;
}());
/* vim: set sw=4 ts=4 et tw=80 : */
