/*
  Copyright (C) 2012-2013 Yusuke Suzuki <utatane.tea@gmail.com>

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
  ARE DISCLAIMED. IN NO EVENT SHALL <COPYRIGHT HOLDER> BE LIABLE FOR ANY
  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*jslint bitwise:true */
/*global module:true, require:true*/
(function () {
    'use strict';

    var Name, Syntax, common, modified;

    Name = 'reduce-branch-jump';
    common = require('../common');
    Syntax = common.Syntax;

    function reduceLast(ary, index) {
        var node, left;
        node = ary[index];
        if (node.type === Syntax.IfStatement) {
            if (!node.alternate) {
                if (node.consequent.type === Syntax.ReturnStatement) {
                    modified = true;
                    left = node.consequent.argument;
                    if (!left) {
                        ary[index] = common.moveLocation(node, {
                            type: Syntax.ReturnStatement,
                            argument: {
                                type: Syntax.SequenceExpression,
                                expressions: [
                                    node.test,
                                    common.SpecialNode.generateUndefined()
                                ]
                            }
                        });
                        return true;
                    }
                    ary[index] = common.moveLocation(node, {
                        type: Syntax.ReturnStatement,
                        argument: {
                            type: Syntax.ConditionalExpression,
                            test: node.test,
                            consequent: left,
                            alternate: common.SpecialNode.generateUndefined()
                        }
                    });
                    return true;
                }
            }
        }
    }

    function reduce(ary, index) {
        var node, sibling, left, right;
        node = ary[index];
        sibling = ary[index + 1];
        if (node.type === Syntax.IfStatement) {
            if (!node.alternate) {
                if (node.consequent.type === Syntax.ReturnStatement && sibling.type === Syntax.ReturnStatement) {
                    // pattern:
                    //     if (cond) return v;
                    //     return v2;
                    modified = true;
                    ary.splice(index, 1);
                    left = node.consequent.argument;
                    right = sibling.argument;
                    if (!left && !right) {
                        ary[index] = common.moveLocation(node, {
                            type: Syntax.ReturnStatement,
                            argument: {
                                type: Syntax.SequenceExpression,
                                expressions: [
                                    node.test,
                                    common.SpecialNode.generateUndefined()
                                ]
                            }
                        });
                        return true;
                    }
                    if (!left) {
                        left = common.SpecialNode.generateUndefined();
                    }
                    if (!right) {
                        right = common.SpecialNode.generateUndefined();
                    }
                    ary[index] = common.moveLocation(node, {
                        type: Syntax.ReturnStatement,
                        argument: {
                            type: Syntax.ConditionalExpression,
                            test: node.test,
                            consequent: left,
                            alternate: right
                        }
                    });
                    return true;
                }
            }
        }
        return false;
    }

    function reduceBranchJump(tree, options) {
        var result;

        result = options.get('destructive', { pathName: Name }) ? tree : common.deepCopy(tree);
        modified = false;

        common.traverse(result, {
            leave: function leave(node, parent) {
                var i;
                switch (node.type) {
                case Syntax.BlockStatement:
                case Syntax.Program:
                    i = 0;
                    while (i < (node.body.length - 1)) {
                        if (!reduce(node.body, i)) {
                            ++i;
                        }
                    }

                    if (common.isFunctionBody(node, parent)) {
                        if (node.body.length > 0) {
                            i = node.body.length - 1;
                            reduceLast(node.body, i);
                        }
                    }
                    break;

                case Syntax.SwitchCase:
                    i = 0;
                    while (i < (node.consequent.length - 1)) {
                        if (!reduce(node.consequent, i)) {
                            ++i;
                        }
                    }
                    break;
                }
            }
        });

        return {
            result: result,
            modified: modified
        };
    }

    reduceBranchJump.passName = Name;
    module.exports = reduceBranchJump;
}());
/* vim: set sw=4 ts=4 et tw=80 : */
