/$ Scan.h

/*
AUSTIN - A Palm OS Security Scanner.
Copyright (C) 2003  @stake, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#ifndef	__SCAN_H__
#define	__SCAN_H__

#define	SCAN_BANNERGRAB		1
#define	SCAN_NOBANNERGRAB	0
#define	SCAN_BANNERTIMEOUT	200

#define	SCAN_HOST			2
#define	SCAN_PORT			1
#define SCAN_PROTOCOL		3
#define SCAN_DATE			4
#define SCAN_TIME			5
#define	SCAN_IMPACT			6
#define	SCAN_NAME			0
#define SCAN_DESC			7
#define	SCAN_URL			8
#define	SCAN_RESULTSIZE		9

// Scanning module
int _Scan_firstvuln; // keep track of whether we've written a result before
string _Scan_resultdb; // the name of the result db
string _Scan_result[SCAN_RESULTSIZE]; // the structure used to store results for writing to DB
int _Scan_ignoreselected; // true if the Selected flag of the Vuln DB should be ignored
string _Scan_overridemethod; // contains the method to override Web Vuln checking with 
pointer _Scan_resformat; // the db format used for storing results
int _Scan_ignoreserver; // if true then only vulns for a specific server will be tested

// used for banner grabbing
string _Scan_bannername, _Scan_bannerstring;

// ports to scan
string _Scan_tcpports, _Scan_udpports;

// flipped to true when initScan() has been executed;
//int _Scan_initScanRun = false;

@doc "Create the array used to hold each scan result before it's written to the DB.";
initScan() {
	_Scan_resformat = Array("is," + SCAN_RESULTSIZE + ",2");
	_Scan_resformat[SCAN_HOST][0] = 1;
	_Scan_resformat[SCAN_HOST][1] = "Host";
	_Scan_resformat[SCAN_PORT][0] = 2;
	_Scan_resformat[SCAN_PORT][1] = "Port";
	_Scan_resformat[SCAN_PROTOCOL][0] = 1;
	_Scan_resformat[SCAN_PROTOCOL][1] = "Protocol";
	_Scan_resformat[SCAN_DATE][0] = 8;
	_Scan_resformat[SCAN_DATE][1] = "Date";
	_Scan_resformat[SCAN_TIME][0] = 9;
	_Scan_resformat[SCAN_TIME][1] = "Time";
	_Scan_resformat[SCAN_IMPACT][0] = 2;
	_Scan_resformat[SCAN_IMPACT][1] = "Impact";
	_Scan_resformat[SCAN_NAME][0] = 1;
	_Scan_resformat[SCAN_NAME][1] = "Name";
	_Scan_resformat[SCAN_DESC][0] = 12;
	_Scan_resformat[SCAN_DESC][1] = "Desc";
	_Scan_resformat[SCAN_URL][0] = 1;
	_Scan_resformat[SCAN_URL][1] = "URL";
//	_Scan_initScanRun = true;
}

@doc "Returns based on the site, port and the current vulnerability.";
buildURL(string site, int port) {
	string p = "";
	
	if (port != 80) p = ":" + port;
	return "http://" + site + p + _Vuln_dbrec[VULN_URI];
}

@doc "Writes a scan result to the database.";
writeScanResult(string host, int port, string protocol,
				int impact, string name, string desc, string url) {
	if (_Scan_firstvuln) {
		if (DBcreate(_Scan_resultdb, _Prefs_dbtype, SCAN_RESULTSIZE, &_Scan_resformat[0][0])) {
			writeln("Couldn't create results database!");
			return false;
		}
	} else if (DBopen(_Scan_resultdb) != 0) {
		writeln("Couldn't open results database!");
		return false;
	}
	_Scan_result[SCAN_HOST] = host;
	_Scan_result[SCAN_PORT] = port;
	_Scan_result[SCAN_PROTOCOL] = protocol;
	_Scan_result[SCAN_DATE] = date(0);
	_Scan_result[SCAN_TIME] = time(0);
	_Scan_result[SCAN_IMPACT] = impact;
	_Scan_result[SCAN_NAME] = name;
	_Scan_result[SCAN_DESC] = desc;
	_Scan_result[SCAN_URL] = url;
	if (DBsetrec(-1, _Scan_result) != 0) return false;
	_Scan_firstvuln = false;
}

@doc "Checks a vulnerability is present on a site. Returns true if it is, false otherwise.";
int checkVuln(string site, int port) {
	// only checkVuln if selected or told to ignore selected flag
	if (!_Scan_ignoreselected && _Vuln_dbrec[VULN_SELECTED] != "1")
		return false;
	
	// if we weren't told to ignore the Server header and the current vuln isn't applicable
	// to the server we're scanning we ignore this vuln
	if (!_Scan_ignoreserver &&
		strstr(_HTTP_server,strlwr(_Vuln_dbrec[VULN_SERVER]),0) == -1)
		return false;

	writeln("Reading page: " + _Vuln_dbrec[VULN_URI]);
	if (!readPage(site, port, _Vuln_dbrec[VULN_URI], _Vuln_dbrec[VULN_METHOD]))
		return false;

	writeln("Checking vuln: " + _Vuln_dbrec[VULN_MATCH]);
	if (strStartsWith(_Vuln_dbrec[VULN_MATCH], VULN_RETCODECHECK)) {
		if (_HTTP_retcode == substr(_Vuln_dbrec[VULN_MATCH],VULN_CHECKLEN,strlen(_Vuln_dbrec[VULN_MATCH])))
			 return true;
		else return false;
	} else if (strStartsWith(_Vuln_dbrec[VULN_MATCH], VULN_MATCHCHECK)) {	
		if (strstr(_HTTP_rawres, substr(_Vuln_dbrec[VULN_MATCH],VULN_CHECKLEN,strlen(_Vuln_dbrec[VULN_MATCH])),0) != -1)
			 return true;
		else return false;
	} else return false;
}

@doc "Returns the name of the scan results database.";
string scanDBName(string name) {
	string t;
	
	t = time(0);
	// ensure we have four digits for time
	while (strlen(t) < 4) t = "0" + t;
	return substr(date(0),4,8) + ":" + t + " " + substr(name,0,10);
}

@doc "Sets variables for the scan database.";
initScanDB(string site) {
	string t;
	
	_Scan_firstvuln = true;
	_Scan_resultdb = scanDBName(site);
}

@doc "URL scans a web site. Returns false if the scan is interrupted.";
int scanSite(string site, int port) {
	setStatus("Web Vuln Scanning: " + site + ":" + port);
	initHTTP();
	if (!getHTTPHeaders(site, port, "/")) return false;
	if (_HTTP_server == "") _Scan_ignoreserver = true;
	else _Scan_ignoreserver = false;
	openVulnDB(VULN_FIRSTOPEN);
	while (readVuln()) {
		if (userHitStop()) return false;
		if (checkVuln(site, port)) {
			writeScanResult(site, port, "TCP", _Vuln_dbrec[VULN_IMPACT], "Web Vuln", 
							_Vuln_dbrec[VULN_DESC], substr(buildURL(site, port), 0, 255));
			openVulnDB(VULN_REOPEN);
		}
	}
	return true;
}

@doc "Read a banner on an open socket.";
string readBanner(int sock) {
	string s;
	int bytes, err;
	
	tcpWrite(sock, _Scan_bannerstring);
	// we call this function directly because we want it to return quickly
	bytes = netSocketReceive(sock, &s, 1000, NET_NOFLAGS, &err);	
	return s;
}

@doc "Given a range s in the form \"123\" or \"123-456\", parseRange will either set lo and hi to \"123\" or to \"123\" and \"456\" respectively.";
parseRange(string s, pointer lo, pointer hi) {
	int index;
	
	index = strstr(s, "-", 0);
	if (index == -1) {
		*lo = *hi = s;
	} else {
		*lo = substr(s, 0, index);
		*hi = substr(s, index+1, 10);
	}
	//FIXME: check for legal ranges
}

@doc "Scans a host and executes the user-specified options. Returns false if the scan is interrupted.";
int scanHost(string ip) {
	string p, banner, portschecked, opentcpports, openudpports;
	int i, j, sock, index, startp, endp;
	
	// TCP Scan
	if (_Prefs_tcpscan) {
		setStatus("TCP Port Scanning: " + ip);
		i = strtok(_Scan_tcpports, &p, ",", 0);
		while (i != -1) {
			parseRange(p, &startp, &endp);
			for (j = startp; j <= endp; j++) {
				if (strstr(portschecked, j, 0) == -1) {
					portschecked = portschecked + j + ",";
				} else {
					continue;
				}
				if (userHitStop()) return false;
				setStatus("Checking " + ip + ":" + j + " (TCP)");
				sock = tcpConnect(ip, j);
				if (sock >= 0) {
					opentcpports = opentcpports + j + ",";
					if (_Prefs_bannergrab) banner = "Banner:\n" + readBanner(sock) + "\n";
					else banner = "";
					tcpClose(sock);
					writeScanResult(ip, j, "TCP", 3, "Open Port", banner, "");
					setStatus(ip + ":" + j + " (TCP) is open");
				}
			}
			i = strtok(_Scan_tcpports, &p, ",", i);
		}
		writeln("opentcpports: " + opentcpports);
	}
	// UDP Scan
	if (_Prefs_udpscan) {
		setStatus("UDP Port Scanning: " + ip);
		portschecked = "";
		i = strtok(_Scan_udpports, &p, ",", 0);
		while (i != -1) {
			parseRange(p, &startp, &endp);
			for (j = startp; j <= endp; j++) {
				if (strstr(portschecked, j, 0) == -1) {
					portschecked = portschecked + j + ",";
				} else {
					continue;
				}
				if (userHitStop()) return false;
				setStatus("Checking " + ip + ":" + j + " (UDP)");
				sock = udpConnect(ip, j);
				if (sock >= 0) {
					banner = readBanner(sock);
					if (banner != "") {
						openudpports = openudpports + j + ",";
						banner = "Banner:\n" + banner + "\n";
						writeScanResult(ip, j, "UDP", 3, "Open Port", banner, "");
						setStatus(ip + ":" + j + " (UDP) is open");
					}
					udpClose(sock);
				}
			}
			i = strtok(_Scan_udpports, &p, ",", i);
		}
		writeln("openudpports: " + openudpports);
	}
	// URL Scan
	if (_Prefs_webvulnscan && opentcpports != "") {
		i = strtok(opentcpports, &p, ",", 0);
		while (i != -1) {
			if (!scanSite(ip, p)) return false;
			i = strtok(opentcpports, &p, ",", i);
		}
	}
	return true;
}

@doc "Scan a specific network or host. Network can be specified as ranges. Returns false if the scan is interrupted.";
int scanNetwork(string net) {
	string ip;
	int i, j=0, a, b, c, d, addr[8];

	// if we're dealing with an IP address or IP address range
	if (isNumericRange(net)) { 
		i = strtok(net, &ip, ".", 0);
		while (i != -1) {
			parseRange(ip, addr+j, addr+j+1);
			i = strtok(net, &ip, ".", i);
			j = j + 2;
		}
		//BUGBUG: check for correct number of bytes
		for (a = addr[0]; a <= addr[1]; a++) {
			for (b = addr[2]; b <= addr[3]; b++) {
				for (c = addr[4]; c <= addr[5]; c++) {
					for (d = addr[6]; d <= addr[7]; d++) {
						ip = a + "." + b + "." + c + "." + d;
						if (!scanHost(ip)) return false;
					}
				}
			}
		}
	} else {
	// we must have a host name
		return (scanHost(net));
	}
	return true;
}

@doc "Scan all hosts specified in the Hosts field, expects a valid Hosts field.";
scanAllNetworks() {
	string net;
	int i, cont = true;
	
	initScanDB(_Prefs_hosts);
	i = strtok(_Prefs_hosts, &net, ",", 0);
	while (i != -1) {
		cont = scanNetwork(net);
		if (! cont) break;
		i = strtok(_Prefs_hosts, &net, ",", i);
	}
	if (cont) setStatus("Scan complete.");
	else setStatus("Scan interrupted.");
}

@doc "Implements the Scheduled Scan option.";
doScheduledScan() {
	int scancount, sleeptime, i;
	string sleepfirst;
	
	if (!getPref(PREFS_SCHEDULE, PREFS_SCANCOUNT, &scancount)) {
		setStatus(ERR_NOSCANCOUNTSPECIFIED);
		return;
	}
	if (!getPref(PREFS_SCHEDULE, PREFS_SLEEPFIRST, &sleepfirst)) {
		setStatus(ERR_NOSLEEPFIRSTSPECIFIED);
		return;
	}
	if (!getPref(PREFS_SCHEDULE, PREFS_SLEEPTIME, &sleeptime)) {
		setStatus(ERR_NOSLEEPTIMESPECIFIED);
		return;
	}
	do {
		if (i > 0) deepsleep(sleeptime);
		else if (sleepfirst == "Y") deepsleep(sleeptime);
		// scanAllNetworks();
		setStatus("Scheduled Scan iteration: " + i);
	} while (i++ < scancount); 
}

@doc "Returns true if all scan params are correctly specified, return false otherwise.";
int readyToScan() {
	if (_Prefs_hosts == "") {
		setStatus(ERR_NOHOSTSSPECIFIED);
		return false;
	}
	// ensure ports to scan are specified
	if (_Prefs_ports == "") {
		getAllPrefs(PREFS_TCPSCAN, ",", &_Scan_tcpports);
		getAllPrefs(PREFS_UDPSCAN, ",", &_Scan_udpports);
		if (_Scan_tcpports == "" && _Prefs_tcpscan) {
				setStatus(ERR_NOTCPPORTSSPECIFIED);
				return false;
		}
		if (_Scan_udpports == "" && _Prefs_udpscan) {
				setStatus(ERR_NOUDPPORTSSPECIFIED);
				return false;
		}
	} else {
		_Scan_tcpports = _Prefs_ports;
		_Scan_udpports = _Prefs_ports;
	}
	// at least one type of scan must be specified
	if (!_Prefs_tcpscan && !_Prefs_udpscan) {
		setStatus(ERR_NOSCANTYPESPECIFIED);
		return false;
	}
	setStatus("");
	return true;
}

@doc "Executed when the user selects the Scan! button.";
doScan() {
	string s;
	
	// ensure we have all the necessary inputs
	if (!readyToScan()) return;
	_Prefs_hosts = stripWhitespace(_Prefs_hosts);
	// ensure that the ports are correctly spec'ed
	_Prefs_ports = stripWhitespace(_Prefs_ports);
	if (!isNumericRange(_Prefs_ports)) {
		setStatus(ERR_INVALIDPORTS);
		return;
	}
	// a Web Vuln Scan implies Banner Grabbing 'cos we will check that the ports really
	// response with HTTP
	// if we're banner grabbing, make sure we know what string to use
	if (_Prefs_bannergrab) {
		if (!getFirstPref(PREFS_BANNER, &_Scan_bannername, &_Scan_bannerstring)) {
			setStatus(ERR_NOBANNERSTRING);
			return;
		}
	}
	// ignore the Vuln DB selected flag if necessary
	if (getPref(PREFS_WEBVULN, PREFS_IGNORESELECT, &s)) _Scan_ignoreselected = true;
	else _Scan_ignoreselected = false;
	// figure out whether to override the Web Vuln HTTP method
	if (!getPref(PREFS_WEBVULN, PREFS_OVERRIDEMETHOD, &_Scan_overridemethod))
		_Scan_overridemethod = "";
	// turn the Scan! button to Stop!
	setButtonString(GUI_SCANBU, "Stop!");
	// figure out whether to scan or schedule scan
	if (!_Prefs_scheduledscan) {
		scanAllNetworks();
	} else {
		doScheduledScan();
	}
	// turn the Stop! button to Scan!
	setButtonString(GUI_SCANBU, "Scan!");
}

#endif
