/$ PrefsDB.h

/*
AUSTIN - A Palm OS Security Scanner.
Copyright (C) 2003  @stake, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

// Handles all Preferences for AUSTIN, which includes storing a user's inputs
// and reading the Preferences database itself

#ifndef	__PREFSDB_H__
#define	__PREFSDB_H__

// AUSTIN Prefs DB names
#define	PREFS_DB				"AUSTIN Prefs DB"	// this is in DB format and exposed to user
#define	PREFS_GUIDB				"AUSTIN GUI DB"		// this is Palm db format and hidden from user

// Format of the GUI prefs
#define	PREFS_GUIPREFSFMT			"2sz6i2"

// Pref Types
#define	PREFS_TCPSCAN			"TCP"
#define	PREFS_UDPSCAN			"UDP"
#define	PREFS_BANNER			"BANNER"
#define PREFS_WEBVULN			"URL"
#define	PREFS_SCHEDULE			"SCHEDULE"
#define	PREFS_MISC				"MISC"

// Pref Names
#define	PREFS_SCANCOUNT			"Scan count"
#define	PREFS_SLEEPFIRST		"Sleep first"
#define	PREFS_SLEEPTIME			"Sleep time"
#define	PREFS_IGNORESELECT		"Ignore selected"
#define	PREFS_OVERRIDEMETHOD	"Override method"
#define	PREFS_DEBUG				"Debug"
#define	PREFS_NETTIMEOUT		"Net Timeout"

// Column numbers for the Prefs DB
#define	PREFS_TYPE			0
#define	PREFS_NAME			1
#define	PREFS_VALUE			2
#define	PREFS_DESC			3
#define	PREFS_SELECTED		4


// linked list heads
pointer _Prefs_tcphead=0,
		_Prefs_udphead=0,
		_Prefs_bannerhead=0,
		_Prefs_webvulnhead=0,
		_Prefs_schedulehead=0,
		_Prefs_mischead=0;

// array to store each preferences record
string _Prefs_dbrec[5];

// indicates whether we're dealing with a HanDBase or a Pilot-DB database
// this is dictated by the type of the Prefs Database
int _Prefs_dbtype;

// GUI prefs (must be kept in this order)
string _Prefs_hosts,
	   _Prefs_ports;
int _Prefs_tcpscan,
	_Prefs_udpscan,
	_Prefs_naslscan,
	_Prefs_webvulnscan,
	_Prefs_scheduledscan,
	_Prefs_bannergrab;

@doc "Takes a pointer to an integer and reverses its boolean value.";
togglePref(pointer p) {
	if (*p) *p = false;
	else *p = true;
}

@doc "Prints the details of preferences record. Used for debugging.";
printPrefsRec() {
	writeln("Type:\t" + _Prefs_dbrec[PREFS_TYPE]);
	writeln("Name:\t" + _Prefs_dbrec[PREFS_NAME]);
	writeln("Value:\t" + _Prefs_dbrec[PREFS_VALUE]);
	writeln("Desc:\t" + _Prefs_dbrec[PREFS_DESC]);
	writeln("Select:\t" + _Prefs_dbrec[PREFS_SELECTED]);
	writeln("");
}

@doc "Returns the appropriate list based on the type passed in.";
getPrefsList(string type) {
	switch (type) {
		case PREFS_TCPSCAN:		return &_Prefs_tcphead;
		case PREFS_UDPSCAN:		return &_Prefs_udphead;
		case PREFS_BANNER:		return &_Prefs_bannerhead;
		case PREFS_WEBVULN:		return &_Prefs_webvulnhead;
		case PREFS_SCHEDULE:	return &_Prefs_schedulehead;
		case PREFS_MISC:		return &_Prefs_mischead;
		default:				return 0;
	}
}

@doc "Add a preferences entry to the appropriate linked list.";
addPref() {
	pointer list, elem;
	
	list = getPrefsList(_Prefs_dbrec[PREFS_TYPE]);
	if (list == 0) {
		writeln("Unknown prefs type: " + _Prefs_dbrec[PREFS_TYPE]);
		return false;
	}
	elem = malloct(1, "ssp");
	elem[0] = _Prefs_dbrec[PREFS_NAME];
	elem[1] = _Prefs_dbrec[PREFS_VALUE];
	elem[2] = *list;
	*list = elem;
}

@doc "Prints a single preferences list, used for debugging.";
printList(string type) {
	pointer elem;
	
	writeln("LIST: " + type);
	elem = getPrefsList(type);
	elem = *elem;
	if (elem == 0) writeln("<empty>");
	while (elem != 0) {
		writeln(elem[0] + "\t" + elem[1]);
		elem = elem[2];
	}
	writeln("");
}

@doc "Prints all preferences lists, used for debugging.";
printAllLists() {
	printList(PREFS_TCPSCAN);
	printList(PREFS_UDPSCAN);
	printList(PREFS_BANNER);
	printList(PREFS_WEBVULN);
	printList(PREFS_SCHEDULE);
	printList(PREFS_MISC);
}

@doc "Reads the parameters saved from the user's last session.";
initGUIPrefs() {
	int vers, i;

	if (dbopen(PREFS_GUIDB)) {
		vers = dbread('i');
		i = dbreadx(&_Prefs_hosts, PREFS_GUIPREFSFMT);
	}
}

@doc "Reads in all preferences and stores selected ones in memory.";
initPrefs() {
	int i, res, totrecs;
	
	if (DBopen(PREFS_DB) != 0) return false;
	totrecs = DBinfo(2, 0);
	_Prefs_dbtype = DBinfo(0, 0);
	for (i = 0; i < totrecs; i++) {
		if (DBgetrec(i, _Prefs_dbrec)) {
			writeln("Error reading prefs rec #" + i);
			break;
		}
		if (_Prefs_dbrec[PREFS_SELECTED] != "1") continue;
		addPref();				
	}
}

@doc "Performs cleanup/saving before AUSTIN exists.";
closePrefs() {
	int i;
	
	dbcreate(PREFS_GUIDB);
	dbwrite(_AUSTIN_version);
	i = dbwritex(&_Prefs_hosts, PREFS_GUIPREFSFMT);
}

@doc "Sets str to the prefs of the corresponding type, concatenating in the case of multiple matching prefs, separating each pref by \",\". Returns true if successful, false if no corresponding prefs are found";
getAllPrefs(string type, string sep, pointer str) {
	pointer elem;
	
	elem = getPrefsList(type);
	elem = *elem;
	*str = "";
	while (elem != 0) {
		if (*str == "") *str = elem[1];
		else *str = *str + sep + elem[1];
		elem = elem[2];
	}
	if (*str != "")	return true;
	else return false;

}

@doc "Sets name & value to the first pref found of the corresponding type. Returns true if successful, false if no corresponding pref are found.";
getFirstPref(string type, pointer name, pointer value) {
	pointer elem;
	
	elem = getPrefsList(type);
	elem = *elem;
	if (elem == 0) return false;
	*name = elem[0];
	*value = elem[1];
	return true;
}

@doc "Sets value to the pref of the corresponding type & name. Returns true if successful, false if no corresponding pref is found.";
getPref(string type, string name, pointer value) {
	pointer elem;
	
	elem = getPrefsList(type);
	elem = *elem;
	while (elem != 0) {
		if(elem[0] == name) {
			*value = elem[1];
			return true;
		}
		elem = elem[2];
	}
	return false;
}

#endif
